define(['FHIRPageableObjectFactory', 'RequestService'], function (FHIRPageableObjectFactory, RequestService) {
    'use strict';

    describe('FHIR Pageable Object Factory', function () {
        var pageableObjectFactory,
            pageableObject,
            requestService,
            fhirResponseObj = {
                data: {
                    link: [
                        {
                            relation: 'self',
                            url: 'http://localhost/fhir/Observation/self'
                        },
                        {
                            relation: 'next',
                            url: 'http://localhost/fhir/Observation/next'
                        }
                    ],
                    total: 10,
                    entry: [
                        { name: 'first' },
                        { name: 'second' },
                        { name: 'third' },
                        { name: 'fourth' },
                        { name: 'fifth' },
                        { name: 'sixth' },
                        { name: 'seventh' },
                        { name: 'eighth' },
                        { name: 'ninth' },
                        { name: 'tenth' }
                    ]
                }
            };

        beforeEach(function () {
            pageableObjectFactory = FHIRPageableObjectFactory(RequestService());
            pageableObject = pageableObjectFactory.createPageableObject();

            spyOn(pageableObject, 'sendRequest');
            spyOn(pageableObject, 'findInArray').and.callThrough();
        });

        describe('Pageable Object', function () {
            describe('constructor', function () {
                it('should have default values for all of the private variables', function () {
                    pageableObject.constructor();

                    expect(pageableObject.getItems()).toEqual([]);
                    expect(pageableObject.getTotal()).toEqual(0);
                    expect(pageableObject.getLinks()).toEqual([]);
                    expect(pageableObject.getSelfLink()).toEqual({});
                });

                it('should initialize the selfLink object to the self link of the FHIR response object', function () {
                    pageableObject.constructor(fhirResponseObj);

                    expect(pageableObject.getSelfLink()).toEqual({
                        relation: 'self',
                        url: 'http://localhost/fhir/Observation/self'
                    });
                });

                it('should initialize the items, total, and links variables based on the FHIR response object', function () {
                    pageableObject.constructor(fhirResponseObj);

                    expect(pageableObject.getTotal()).toEqual(10);
                    expect(pageableObject.getItems()).toEqual(fhirResponseObj.data.entry);
                    expect(pageableObject.getLinks()).toEqual(fhirResponseObj.data.link);
                });
            });

            describe('update function', function () {
                it('should set all of the variables to default values if no object is passed in', function () {
                    pageableObject.update();

                    expect(pageableObject.getTotal()).toEqual(0);
                    expect(pageableObject.getItems()).toEqual([]);
                    expect(pageableObject.getLinks()).toEqual([]);
                });

                it('should set all of the variables to default values if an object is passed in without a data object', function () {
                    pageableObject.update({});

                    expect(pageableObject.getTotal()).toEqual(0);
                    expect(pageableObject.getItems()).toEqual([]);
                    expect(pageableObject.getLinks()).toEqual([]);
                });

                it('should set all of the variables based on the FHIR response object passed in', function () {
                    pageableObject.update(fhirResponseObj);

                    expect(pageableObject.getTotal()).toEqual(10);
                    expect(pageableObject.getItems()).toEqual(fhirResponseObj.data.entry);
                    expect(pageableObject.getLinks()).toEqual(fhirResponseObj.data.link);
                });
            });

            describe('getLinkByRelation function', function () {
                beforeEach(function () {
                    pageableObject.constructor(fhirResponseObj);
                });

                it('should search the list of links passed into the function', function () {
                    var link = pageableObject.getLinkByRelation('self', []);

                    expect(link).toBeFalsy();
                });

                it('should search the links of list from the PageableObject by default', function () {
                    var link = pageableObject.getLinkByRelation('self');

                    expect(link).toBeTruthy();
                });

                it('should return a link object when a link is found in the links list', function () {
                    var link = pageableObject.getLinkByRelation('self', [
                        {
                            relation: 'self',
                            url: 'http://localhost/fhir/Observation/self'
                        }
                    ]);

                    expect(link).toEqual({
                        relation: 'self',
                        url: 'http://localhost/fhir/Observation/self'
                    });
                });

                it('should return undefined when a link is not found in the links list', function () {
                    var link = pageableObject.getLinkByRelation('next', [
                        {
                            relation: 'self',
                            url: 'http://localhost/fhir/Observation/self'
                        }
                    ]);

                    expect(link).toEqual(undefined);
                });

                it('should gracefully skip link objects without relation attributes', function () {
                    var link = pageableObject.getLinkByRelation('self', [
                        {
                            url: 'http://localhost/fhir/someOtherUrl'
                        },
                        {
                            relation: 'self',
                            url: 'http://localhost/fhir/Observation/self'
                        }
                    ]);

                    expect(link).toBeTruthy();
                });
            });

            describe('next function', function () {
                beforeEach(function () {
                    spyOn(pageableObject, 'getNextLink');
                    spyOn(pageableObject, 'changePage');
                });

                it('should throw an error if the next link is not found', function () {
                    pageableObject.getNextLink.and.callFake(function () {
                        return undefined;
                    });

                    try {
                        pageableObject.next();
                        fail('Error should have been thrown');
                    } catch (e) {
                        expect(e).toBeTruthy();
                        expect(e.message).toEqual('Next link not found: end of list.');
                    };
                });

                it('should throw an error if the next link doesn\'t have a URL', function () {
                    pageableObject.getNextLink.and.callFake(function () {
                        return {};
                    });

                    try {
                        pageableObject.next();
                        fail('Error should have been thrown');
                    } catch (e) {
                        expect(e).toBeTruthy();
                        expect(e.message).toEqual('Next link not found: end of list.');
                    };
                });

                it('should call changePage if the next link is valid', function () {
                    var url = 'http://localhost/fhir/Observation/next';

                    pageableObject.getNextLink.and.callFake(function () {
                        return {
                            url: url
                        };
                    });

                    pageableObject.next();

                    expect(pageableObject.changePage).toHaveBeenCalledWith(url);
                });
            });

            describe('prev function', function () {
                beforeEach(function () {
                    spyOn(pageableObject, 'getPrevLink');
                    spyOn(pageableObject, 'changePage');
                });

                it('should throw an error if the previous link is not found', function () {
                    pageableObject.getPrevLink.and.callFake(function () {
                        return undefined;
                    });

                    try {
                        pageableObject.prev();
                        fail('Error should have been thrown');
                    } catch (e) {
                        expect(e).toBeTruthy();
                        expect(e.message).toEqual('Previous link not found: beginning of list.');
                    };
                });

                it('should throw an error if the previous link doesn\'t have a URL', function () {
                    pageableObject.getPrevLink.and.callFake(function () {
                        return {};
                    });

                    try {
                        pageableObject.prev();
                        fail('Error should have been thrown');
                    } catch (e) {
                        expect(e).toBeTruthy();
                        expect(e.message).toEqual('Previous link not found: beginning of list.');
                    };
                });

                it('should call changePage if the next link is valid', function () {
                    var url = 'http://localhost/fhir/Observation/prev';

                    pageableObject.getPrevLink.and.callFake(function () {
                        return {
                            url: url
                        };
                    });

                    pageableObject.prev();

                    expect(pageableObject.changePage).toHaveBeenCalledWith(url);
                });
            });

            describe('getNextLink function', function () {
                beforeEach(function () {
                    spyOn(pageableObject, 'getLinkByRelation');
                });

                it('should call getLinkByRelation for the next link', function () {
                    pageableObject.getNextLink();

                    expect(pageableObject.getLinkByRelation).toHaveBeenCalledWith('next', undefined);
                });

                it('should call getLinkByRelation for the next link in the passed in links array', function () {
                    var links = [];

                    pageableObject.getNextLink(links);

                    expect(pageableObject.getLinkByRelation).toHaveBeenCalledWith('next', links);
                });
            });

            describe('getPrevLink function', function () {
                beforeEach(function () {
                    spyOn(pageableObject, 'getLinkByRelation');
                });

                it('should call getLinkByRelation for the previous link', function () {
                    pageableObject.getPrevLink();

                    expect(pageableObject.getLinkByRelation).toHaveBeenCalledWith('previous', undefined);
                });

                it('should call getLinkByRelation for the previous link in the passed in links array', function () {
                    var links = [];

                    pageableObject.getPrevLink(links);

                    expect(pageableObject.getLinkByRelation).toHaveBeenCalledWith('previous', links);
                });
            });

            describe('changePage function', function () {
                var responseObj = {
                    resourceType: 'Observation'
                };

                beforeEach(function () {
                    spyOn(pageableObject, 'update');
                    pageableObject.sendRequest.and.callFake(function () {
                        return {
                            then: function (callback) {
                                if (typeof callback === 'function') {
                                    callback(responseObj);
                                }
                            }
                        }
                    });
                });

                it('should throw an error if no url is passed in', function () {
                    try {
                        pageableObject.changePage();
                        fail('Error should have been thrown');
                    } catch (e) {
                        expect(e).toBeTruthy();
                        expect(e.message).toEqual('URL must be supplied.  Usage: changePage(url)');
                    }
                });

                it('should send a GET request and call the update function with the results', function () {
                    var url = 'http://localhost/fhir/Observation';

                    pageableObject.changePage(url);

                    expect(pageableObject.sendRequest).toHaveBeenCalledWith({
                        method: 'GET',
                        url: url
                    });

                    expect(pageableObject.update).toHaveBeenCalledWith(responseObj);
                });
            });

            describe('getAllItems function', function () {
                var thenFunction = function (callback) {
                    return callback();
                };

                beforeEach(function () {
                    pageableObject.sendRequest.and.callFake(function () {
                        return {
                            then: function (callback) {
                                if (typeof callback === 'function') {
                                    return thenFunction(callback);
                                }
                            }
                        }
                    });
                });

                it('should make a GET request for the maximum number of items per page (50)', function () {
                    var selfLink = pageableObject.getSelfLink();

                    pageableObject.getAllItems();

                    expect(pageableObject.sendRequest).toHaveBeenCalledWith({
                        method: 'GET',
                        url: selfLink.url,
                        params: {
                            _count: 50
                        }
                    });
                });

                it('should return an empty list if the response from the server is empty', function () {
                    var allItems = pageableObject.getAllItems();

                    expect(allItems).toEqual([]);
                });

                it('should return an empty list if the data object in the response is empty', function () {
                    thenFunction = function (callback) {
                        return callback({ data: undefined });
                    };

                    var allItems = pageableObject.getAllItems();

                    expect(allItems).toEqual([]);
                });

                it('should return an empty list if the link array in the response is empty', function () {
                    thenFunction = function (callback) {
                        return callback({ data: { link: undefined } });
                    };

                    var allItems = pageableObject.getAllItems();

                    expect(allItems).toEqual([]);
                });

                it('should return the complete list if there\'s only one page of items', function () {
                    var items = [
                        { name: 'first' },
                        { name: 'second' },
                        { name: 'third' },
                        { name: 'fourth' },
                        { name: 'fifth' },
                        { name: 'sixth' },
                        { name: 'seventh' },
                        { name: 'eight' },
                        { name: 'ninth' },
                        { name: 'tenth' }
                    ]

                    thenFunction = function (callback) {
                        return callback({
                            data: {
                                link: [],
                                entry: items
                            }
                        });
                    };

                    var allItems = pageableObject.getAllItems();

                    expect(allItems).toEqual(items);
                });

                it('should return the complete list if there are multiple pages of items', function () {
                    var currPage = 0;
                    var numPages = 3;
                    var numPerPage = 10;
                    var items = [
                        { name: 'first' },
                        { name: 'second' },
                        { name: 'third' },
                        { name: 'fourth' },
                        { name: 'fifth' },
                        { name: 'sixth' },
                        { name: 'seventh' },
                        { name: 'eight' },
                        { name: 'ninth' },
                        { name: 'tenth' },
                        { name: 'eleventh' },
                        { name: 'twelfth' },
                        { name: 'thirteenth' },
                        { name: 'fourteenth' },
                        { name: 'fifteenth' },
                        { name: 'sixteenth' },
                        { name: 'seventeenth' },
                        { name: 'eighteenth' },
                        { name: 'ninteenth' },
                        { name: 'twentieth' },
                        { name: 'twenty-first' },
                        { name: 'twenty-second' },
                        { name: 'twenty-third' },
                        { name: 'twenty-fourth' },
                        { name: 'twenty-fifth' }
                    ];

                    thenFunction = function (callback) {
                        var response = { data: {} };

                        response.data.entry = items.slice(currPage * numPerPage, numPerPage + (numPerPage * currPage));

                        currPage++;

                        if (currPage < numPages) {
                            response.data.link = [
                                {
                                    relation: 'next',
                                    url: 'http://localhost/fhir/Observation/next'
                                }
                            ];
                        }

                        return callback(response);
                    }

                    var allItems = pageableObject.getAllItems();

                    expect(allItems).toEqual(items)
                });
            });
        });
    });
});